/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.wicket.page;


/**
 * Page manager.
 * 
 * @author Matej Knopp
 */
public interface IPageManager
{
	/**
	 * 
	 * @return the page manager context
	 */
	IPageManagerContext getContext();

	/**
	 * Retrieve page instance with given id.
	 * 
	 * @param id
	 *      the id of the page to load
	 * @return page instance or <code>null</code>
	 * @throws CouldNotLockPageException if the page is already locked by another thread
	 * and the lock cannot be acquired for some timeout
	 */
	IManageablePage getPage(int id) throws CouldNotLockPageException;

	/**
	 * Removes a page from the {@link org.apache.wicket.pageStore.IPageStore} and
	 * {@link org.apache.wicket.pageStore.IDataStore}. Any attempt to access it later
	 * will lead to {@link org.apache.wicket.protocol.http.PageExpiredException}
	 *
	 * @param page The page instance to remove from the stores
	 */
	void removePage(IManageablePage page);

	/**
	 * Marks page as changed.
	 * <p><strong>Note:</strong>Only stateful pages are stored.</p>
	 * 
	 * @param page
	 *      the page that should be stored in the page stores at the end of the request.
	 * @throws CouldNotLockPageException if the page is already locked by another thread
	 * and the lock cannot be acquired for some timeout
	 */
	void touchPage(IManageablePage page);

	/**
	 * Marks page as non-changed.
	 * Could be used in Ajax requests to avoid storing the page if no changes have happened.
	 *
	 * @param page
	 *      the page that should <strong>not</strong> be stored in the page stores at the end of the request.
	 * @throws CouldNotLockPageException if the page is already locked by another thread
	 * and the lock cannot be acquired for some timeout
	 */
	void untouchPage(IManageablePage page);

	/**
	 * Returns whether this manager supports versioning. Managers that support versioning must store
	 * page snapshots.
	 * 
	 * @return whether this page manager supports versioning
	 */
	boolean supportsVersioning();

	/**
	 * End the current request.
	 */
	default void endRequest() {
	}
	
	/**
	 * Commits the changes to external storage if the manager uses it.
	 * 
	 * Should also detach all pages that were touched during this request.
	 */
	void commitRequest();

	/**
	 * Invoked when new session has been created.
	 */
	void newSessionCreated();

	/**
	 * Clears all data for the current session.
	 */
	void clear();

	/**
	 * Destroy the page manager.
	 */
	void destroy();
}
